<?php

namespace App\Http\Controllers;

use App\Exports\ProposalExport;
use App\Models\BankAccount;
use App\Models\ActivityLog;
use App\Models\Customer;
use App\Models\CustomField;
use App\Models\Invoice;
use App\Models\InvoiceProduct;
use App\Models\Milestone;
use App\Models\Products;
use App\Models\ProductService;
use App\Models\ProductServiceCategory;
use App\Models\Proposal;
use App\Models\ProposalProduct;
use App\Models\StockReport;
use App\Models\Task;
use App\Models\User;
use App\Models\Utility;
use App\Model\Tax;
use Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Log;

class ProposalController extends Controller
{
    public function __construct()
    {
        // Constructor remains empty as in the original
    }

    public function index(Request $request)
    {
        if (\Auth::user()->can('manage proposal')) {
            $customer = Customer::where('created_by', '=', \Auth::user()->creatorId())->get()->pluck('name', 'id');
            $customer->prepend('All', '');

            $status = Proposal::$statues;

            $query = Proposal::where('created_by', '=', \Auth::user()->creatorId());

            if (!empty($request->customer)) {
                $query->where('id', '=', $request->customer);
            }
            if (!empty($request->issue_date)) {
                $date_range = explode('to', $request->issue_date);
                $query->whereBetween('issue_date', $date_range);
            }
            if (!empty($request->status)) {
                $query->where('status', '=', $request->status);
            }
            $proposals = $query->with(['category'])->get();

            return view('proposal.index', compact('proposals', 'customer', 'status'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }

    public function create($customerId = 0)
    {
        if (\Auth::user()->can('create proposal')) {
            $customFields = CustomField::where('created_by', \Auth::user()->creatorId())
                ->where('module', 'proposal')
                ->get();
    
            $proposal_number = \Auth::user()->proposalNumberFormat($this->proposalNumber());
            $customers = Customer::where('created_by', \Auth::user()->creatorId())
                ->get()
                ->pluck('name', 'id');
            $customers->prepend('Select Customer', '');
    
            $category = ProductServiceCategory::where('created_by', \Auth::user()->creatorId())
                ->where('type', 'income')
                ->get()
                ->pluck('name', 'id');
            $category->prepend('Select Category', '');
    
            $product_services = ProductService::where('created_by', \Auth::user()->creatorId())
                ->get()
                ->pluck('name', 'id');
            $product_services->prepend('--', '');
    
            $customer = $customerId ? Customer::find($customerId) : null;
            $settings_data = Utility::settingsById(\Auth::user()->creatorId());
    
            $company_state = strtolower(trim($settings_data['company_state'] ?? ''));
            $billing_state = $customer ? strtolower(trim($customer->billing_state ?? '')) : '';
            $tax_type = strtolower($settings_data['tax_type'] ?? 'gst'); // default to gst if not set
    
            return view('proposal.create', compact(
                'customers',
                'proposal_number',
                'product_services',
                'category',
                'customFields',
                'customerId',
                'customer',
                'settings_data',
                'company_state',
                'billing_state',
                'tax_type'
            ));
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function customer(Request $request)
    {
        $customer = Customer::where('id', '=', $request->id)->first();

        if ($customer) {
            $billingState = strtolower(trim($customer->billing_state ?? ''));
            $companyState = strtolower(trim(\Auth::user()->settings_data['company_state'] ?? 'defaultstate'));

            \Log::info('State Comparison:', [
                'billing_state' => $billingState,
                'company_state' => $companyState,
                'match' => $billingState === $companyState,
                'raw_billing_state' => $customer->billing_state,
                'customer_id' => $request->id
            ]);

            $customerDetailView = view('proposal.customer_detail', compact('customer'))->render();

            return response()->json([
                'view' => $customerDetailView,
                'state' => $billingState,
                'city' => $customer->billing_city,
                'country' => $customer->billing_country,
                'address' => $customer->billing_address,
                'postal_code' => $customer->billing_zip,
            ]);
        }

        return response()->json(['error' => 'Customer not found'], 404);
    }

   public function edit($ids)
{
    if (\Auth::user()->can('edit proposal')) {
        try {
            $id = Crypt::decrypt($ids);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', __('Proposal Not Found.'));
        }

        $proposal = Proposal::find($id);
        if ($proposal->created_by != \Auth::user()->creatorId()) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }

        $proposal_number = \Auth::user()->proposalNumberFormat($proposal->proposal_id);
        $customers = Customer::where('created_by', \Auth::user()->creatorId())->pluck('name', 'id');
        $category = ProductServiceCategory::where('created_by', \Auth::user()->creatorId())
            ->where('type', 'income')->pluck('name', 'id');
        $category->prepend('Select Category', '');
        $product_services = ProductService::where('created_by', \Auth::user()->creatorId())->pluck('name', 'id');
        $proposal->customField = CustomField::getData($proposal, 'proposal');
        $customFields = CustomField::where('created_by', \Auth::user()->creatorId())
            ->where('module', 'proposal')->get();

        // ✅ Correct settings
        $settings = Utility::settingsById(\Auth::user()->creatorId());
        $tax_type = strtolower($settings['tax_type'] ?? 'gst');
        $company_state = strtolower($settings['company_state'] ?? '');

        $items = [];
        foreach ($proposal->items as $proposalItem) {
            $itemAmount = $proposalItem->quantity * $proposalItem->price;
            $taxRate = $proposalItem->tax ? $this->getTaxRate($proposalItem->tax) : 0;
            $proposalItem->itemAmount = $itemAmount;
            $proposalItem->taxRate = $taxRate;

            if ($tax_type === 'vat') {
                $proposalItem->vatAmount = Utility::taxRate(
                    $taxRate,
                    $proposalItem->price,
                    $proposalItem->quantity,
                    $proposalItem->discount
                );
            }

            $items[] = $proposalItem;
        }

        $customer = Customer::find($proposal->customer_id);
        $customerId = $proposal->customer_id;

        return view('proposal.edit', compact(
            'customers',
            'product_services',
            'proposal',
            'proposal_number',
            'category',
            'customFields',
            'items',
            'customer',
            'customerId',
            'tax_type',
            'company_state' // ✅ pass to view
        ));
    } else {
        return redirect()->back()->with('error', __('Permission denied.'));
    }
}

    

    // Helper method to get total tax rate
    protected function getTaxRate($taxIds)
    {
        if (empty($taxIds) || $taxIds == 0) return 0;
        $taxData = Utility::getTaxData(); // Use Utility::getTaxData() as in your model
        $totalRate = 0;
        $taxArr = explode(',', $taxIds); // Split comma-separated tax_ids
        foreach ($taxArr as $taxId) {
            $taxId = trim($taxId); // Remove any whitespace
            if (!empty($taxData[$taxId]['rate'])) {
                $totalRate += floatval(str_replace('%', '', $taxData[$taxId]['rate']));
            }
        }
        return $totalRate;
    }
    public function product(Request $request)
    {
        $data['product'] = $product = ProductService::find($request->product_id);
        $data['unit'] = (!empty($product->unit)) ? $product->unit->name : '';
        $data['taxRate'] = $taxRate = !empty($product->tax_id) ? $product->taxRate($product->tax_id) : 0;
        $data['taxes'] = !empty($product->tax_id) ? $product->tax($product->tax_id) : 0;
        $data['expense_chartaccount_code'] = $product->expense_chartaccount_code?? null; 
        $salePrice = $product->sale_price;
        $quantity = 1;
        $discount = 0;
        $baseAmount = ($salePrice * $quantity) - $discount;
        $taxPrice = ($taxRate / 100) * $baseAmount;
        $data['totalAmount'] = $baseAmount;
        return json_encode($data);
    }

    public function store(Request $request)
    {
        if (\Auth::user()->can('create proposal')) {
            $validator = \Validator::make(
                $request->all(),
                [
                    'customer_id' => 'required',
                    'issue_date' => 'required',
                    'category_id' => 'required',
                    'items' => 'required',
                ]
            );
    
            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }
    
            $proposal = new Proposal();
            $proposal->proposal_id = $this->proposalNumber();
            $proposal->customer_id = $request->customer_id;
            $proposal->status = 0;
            $proposal->issue_date = $request->issue_date;
            $proposal->category_id = $request->category_id;
            $proposal->created_by = \Auth::user()->creatorId();
            $proposal->save();
    
            CustomField::saveData($proposal, $request->customField);
    
            $products = $request->items;
            $totalQuantity = 0;
            $totalRate = 0;
            $totalDiscount = 0;
            $taxesData = [];
    
            $settings = Utility::settingsById(\Auth::user()->creatorId());
            $taxType = strtolower($settings['tax_type'] ?? 'gst'); // vat or gst
    
            $customer = Customer::find($request->customer_id);
            $companyState = strtolower(trim($settings['company_state'] ?? ''));
            $billingState = strtolower(trim($customer->billing_state ?? ''));
            $isSameState = !empty($companyState) && !empty($billingState) && $companyState === $billingState;
    
            Log::info('State Comparison', [
                'billing_state' => $billingState,
                'company_state' => $companyState,
                'match' => $isSameState,
                'tax_type' => $taxType
            ]);
    
            foreach ($products as $item) {
                $proposalProduct = new ProposalProduct();
                $proposalProduct->proposal_id = $proposal->id;
                $proposalProduct->product_id = $item['item'];
                $proposalProduct->quantity = $item['quantity'];
                $proposalProduct->discount = $item['discount'];
                $proposalProduct->price = $item['price'];
                $proposalProduct->description = $item['description'];
    
                if (!empty($item['item'])) {
                    $productId = $item['item'];
                    $proposalProduct->tax = ProductService::tax_id($productId); // Store tax_id
                    $taxRate = $this->getTaxRate($proposalProduct->tax); // Get tax rate
                    $itemTaxPrice = Utility::taxRate($taxRate, $item['price'], $item['quantity'], $item['discount']);
    
                    $totalQuantity += $item['quantity'];
                    $totalRate += $item['price'] * $item['quantity'];
                    $totalDiscount += $item['discount'];
    
                    if (!empty($itemTaxPrice)) {
                        if ($taxType === 'VAT') {
                            // Apply VAT as a single tax
                            $taxesData['VAT'] = ($taxesData['VAT'] ?? 0) + $itemTaxPrice;
                        } else {
                            // Apply GST (CGST/SGST or IGST)
                            if ($isSameState) {
                                $halfTaxPrice = $itemTaxPrice / 2;
                                $taxesData['CGST'] = ($taxesData['CGST'] ?? 0) + $halfTaxPrice;
                                $taxesData['SGST'] = ($taxesData['SGST'] ?? 0) + $halfTaxPrice;
                            } else {
                                $taxesData['IGST'] = ($taxesData['IGST'] ?? 0) + $itemTaxPrice;
                            }
                        }
                    }
                    Log::info('Tax Data', [
                        'tax_id' => $proposalProduct->tax,
                        'tax_rate' => $taxRate,
                        'item_tax_price' => $itemTaxPrice,
                        'tax_type' => $taxType
                    ]);
                }
    
                $proposalProduct->save();
                Log::info('Saved Proposal Product', ['product' => $proposalProduct->toArray()]);
            }
    
            $setting = Utility::settings(\Auth::user()->creatorId());
            $customer = Customer::find($proposal->customer_id);
            $proposalNotificationArr = [
                'proposal_number' => \Auth::user()->proposalNumberFormat($proposal->proposal_id),
                'user_name' => \Auth::user()->name,
                'customer_name' => $customer->name,
                'proposal_issue_date' => $proposal->issue_date,
            ];
    
            if (!empty($setting['twilio_proposal_notification']) && $setting['twilio_proposal_notification'] == 1) {
                Utility::send_twilio_msg($customer->contact, 'new_proposal', $proposalNotificationArr);
            }
    
            return redirect()->route('proposal.index', $proposal->id)->with('success', __('Proposal successfully created.'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
    public function update(Request $request, Proposal $proposal)
    {
        if (\Auth::user()->can('edit proposal')) {
            if ($proposal->created_by != \Auth::user()->creatorId()) {
                return redirect()->back()->with('error', __('Permission denied.'));
            }
    
            $validator = \Validator::make($request->all(), [
                'customer_id' => 'required',
                'issue_date' => 'required',
                'category_id' => 'required',
                'items' => 'required',
            ]);
    
            if ($validator->fails()) {
                return redirect()->route('proposal.index')->with('error', $validator->errors()->first());
            }
    
            $proposal->customer_id = $request->customer_id;
            $proposal->issue_date = $request->issue_date;
            $proposal->category_id = $request->category_id;
            $proposal->save();
            CustomField::saveData($proposal, $request->customField);
    
            $products = $request->items;
            $settings = Utility::settingsById(\Auth::user()->creatorId());
            $tax_type = strtolower($settings['tax_type'] ?? 'gst');
            $company_state = strtolower(trim($settings['company_state'] ?? ''));
    
            $customer = Customer::find($request->customer_id);
            $billing_state = strtolower(trim($customer->billing_state ?? ''));
            $isSameState = (!empty($company_state) && !empty($billing_state) && $company_state === $billing_state);
    
            Log::info('State Comparison', [
                'billing_state' => $billing_state,
                'company_state' => $company_state,
                'match' => $isSameState,
                'raw_billing_state' => $customer->billing_state,
                'customer_id' => $customer->id
            ]);
    
            $totalQuantity = 0;
            $totalRate = 0;
            $totalDiscount = 0;
            $taxesData = [];
    
            foreach ($products as $productData) {
                $proposalProduct = ProposalProduct::find($productData['id']);
                if (!$proposalProduct) {
                    $proposalProduct = new ProposalProduct();
                    $proposalProduct->proposal_id = $proposal->id;
                }
    
                $proposalProduct->product_id = $productData['item'] ?? null;
                $proposalProduct->quantity = $productData['quantity'];
                $proposalProduct->discount = $productData['discount'];
                $proposalProduct->price = $productData['price'];
                $proposalProduct->description = $productData['description'];
    
                if (!empty($proposalProduct->product_id)) {
                    $proposalProduct->tax = ProductService::tax_id($proposalProduct->product_id);
                    $taxRate = $this->getTaxRate($proposalProduct->tax);
                    $itemTaxPrice = Utility::taxRate($taxRate, $proposalProduct->price, $proposalProduct->quantity, $proposalProduct->discount);
    
                    $totalQuantity += $proposalProduct->quantity;
                    $totalRate += $proposalProduct->price * $proposalProduct->quantity;
                    $totalDiscount += $proposalProduct->discount;
    
                    if (!empty($itemTaxPrice)) {
                        if ($tax_type === 'vat') {
                            $taxesData['VAT'] = ($taxesData['VAT'] ?? 0) + $itemTaxPrice;
                        } else {
                            if ($isSameState) {
                                $halfTaxPrice = $itemTaxPrice / 2;
                                $taxesData['CGST'] = ($taxesData['CGST'] ?? 0) + $halfTaxPrice;
                                $taxesData['SGST'] = ($taxesData['SGST'] ?? 0) + $halfTaxPrice;
                            } else {
                                $taxesData['IGST'] = ($taxesData['IGST'] ?? 0) + $itemTaxPrice;
                            }
                        }
                    }
    
                    Log::info('Tax Data', [
                        'tax_id' => $proposalProduct->tax,
                        'tax_rate' => $taxRate,
                        'item_tax_price' => $itemTaxPrice,
                        'tax_type' => strtoupper($tax_type)
                    ]);
                }
    
                $proposalProduct->save();
                Log::info('Updated Proposal Product', ['product' => $proposalProduct->toArray()]);
            }
    
            return redirect()->route('proposal.index')->with('success', __('Proposal successfully updated.'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
    
    

    public function proposal($proposal_id)
    {
        $settings = Utility::settings();
    
        try {
            $proposalId = Crypt::decrypt($proposal_id);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Something went wrong.'));
        }
    
        $proposal = Proposal::with('items.product')->find($proposalId);
    
        if (!$proposal) {
            return redirect()->back()->with('error', __('Proposal not found.'));
        }
    
        // 🔥 Load settings for the proposal creator
        $data = DB::table('settings')->where('created_by', '=', $proposal->created_by)->get();
        foreach ($data as $row) {
            $settings[$row->name] = $row->value;
        }
    
        // ✅ Tax type (vat or gst)
        $tax_type = strtolower($settings['tax_type'] ?? 'gst');
    
        $customer = $proposal->customer;
        $items = [];
        $totalQuantity = 0;
        $totalRate = 0;
        $totalDiscount = 0;
        $taxesData = [];
    
        $companyState = strtolower(trim(Utility::settingsById($proposal->created_by)['company_state'] ?? ''));
        $billingState = strtolower(trim($customer->billing_state ?? ''));
        $isSameState = (!empty($companyState) && !empty($billingState) && $companyState === $billingState);
    
        foreach ($proposal->items as $product) {
            $productService = $product->product;
            $item = new \stdClass();
            $item->name = $productService->name ?? '';
            $item->quantity = $product->quantity;
            $item->unit = $productService->unit_id ?? '';
            $item->price = $product->price;
            $item->discount = $product->discount;
            $item->description = $product->description;
    
            $totalQuantity += $item->quantity;
            $totalRate += $item->price * $item->quantity;
            $totalDiscount += $item->discount;
    
            $taxes = [];
            $baseTaxAmount = 0;
    
            if (!empty($productService) && $productService->tax_id) {
                $taxRates = $productService->tax($productService->tax_id);
    
                foreach ($taxRates as $tax) {
                    $rate = floatval(str_replace('%', '', $tax->rate ?? 0));
                    $taxAmount = Utility::taxRate($rate, $item->price, $item->quantity, $item->discount);
                    $baseTaxAmount += $taxAmount;
    
                    Log::info('Tax calculation', [
                        'product_id' => $productService->id,
                        'tax_name' => $tax->name,
                        'rate' => $rate,
                        'taxAmount' => $taxAmount,
                    ]);
                }
    
                if ($tax_type === 'vat') {
                    // ✅ VAT - no state split
                    $taxes[] = [
                        'name' => 'VAT',
                        'rate' => $taxRates[0]->rate ?? '0%',
                        'price' => Utility::priceFormat($settings, $baseTaxAmount),
                        'tax_price' => $baseTaxAmount,
                    ];
                    $taxesData['VAT'] = ($taxesData['VAT'] ?? 0) + $baseTaxAmount;
                } else {
                    // ✅ GST - state based split
                    if ($isSameState) {
                        $halfTax = $baseTaxAmount / 2;
                        $halfRate = ($taxRates[0]->rate ?? 0) / 2;
    
                        $taxes[] = [
                            'name' => 'CGST',
                            'rate' => $halfRate . '%',
                            'price' => Utility::priceFormat($settings, $halfTax),
                            'tax_price' => $halfTax,
                        ];
                        $taxes[] = [
                            'name' => 'SGST',
                            'rate' => $halfRate . '%',
                            'price' => Utility::priceFormat($settings, $halfTax),
                            'tax_price' => $halfTax,
                        ];
                        $taxesData['CGST'] = ($taxesData['CGST'] ?? 0) + $halfTax;
                        $taxesData['SGST'] = ($taxesData['SGST'] ?? 0) + $halfTax;
                    } else {
                        $taxes[] = [
                            'name' => 'IGST',
                            'rate' => $taxRates[0]->rate ?? '0%',
                            'price' => Utility::priceFormat($settings, $baseTaxAmount),
                            'tax_price' => $baseTaxAmount,
                        ];
                        $taxesData['IGST'] = ($taxesData['IGST'] ?? 0) + $baseTaxAmount;
                    }
                }
            }
    
            $item->itemTax = $taxes;
            $items[] = $item;
        }
    
        $proposal->itemData = $items;
        $proposal->totalQuantity = $totalQuantity;
        $proposal->totalRate = $totalRate;
        $proposal->totalDiscount = $totalDiscount;
        $proposal->taxesData = $taxesData;
        $proposal->customField = CustomField::getData($proposal, 'proposal');
    
        $accounts = BankAccount::where('created_by', $proposal->created_by)->get();
        $creatorId = optional(\Auth::user())->creatorId() ?? $proposal->created_by;

$customFields = CustomField::where('created_by', $creatorId)
    ->where('module', 'proposal')->get();

    
        $logo = asset(Storage::url('uploads/logo/'));
        $settings_data = Utility::settingsById($proposal->created_by);
        $company_logo = $settings_data['company_logo_dark'] ?? Utility::getValByName('company_logo_dark');
        $proposal_logo = $settings_data['proposal_logo'];
        $img = isset($proposal_logo) && !empty($proposal_logo)
            ? Utility::get_file('proposal_logo/') . $proposal_logo
            : asset($logo . '/' . ($company_logo ?: 'logo-dark.png'));
    
        $color = '#' . ($settings['proposal_color'] ?? 'ffffff');
        $fontColor = Utility::getFontColor($color);
    
        return view('proposal.templates.' . $settings['proposal_template'], compact(
            'proposal', 'color', 'settings', 'customer', 'img', 'fontColor', 'customFields', 'accounts'
        ));
    }
    

    function proposalNumber()
    {
        $latest = Proposal::where('created_by', '=', \Auth::user()->creatorId())->latest()->first();
        return $latest ? $latest->proposal_id + 1 : 1;
    }

    public function show($ids)
    {
        if (\Auth::user()->can('show proposal')) {
            try {
                $id = Crypt::decrypt($ids);
            } catch (\Throwable $th) {
                return redirect()->back()->with('error', __('Proposal Not Found.'));
            }
    
            $proposal = Proposal::with(['items.product.unit'])->find($id);
    
            if ($proposal->created_by != \Auth::user()->creatorId()) {
                return redirect()->back()->with('error', __('Permission denied.'));
            }
    
            $customer = $proposal->customer;
            $iteams = $proposal->items;
            $status = Proposal::$statues;
    
            $proposal->customField = CustomField::getData($proposal, 'proposal');
            $customFields = CustomField::where('created_by', \Auth::user()->creatorId())
                ->where('module', 'proposal')->get();
    
            // 🔥 Get tax type and company state
            $settings = Utility::settingsById(\Auth::user()->creatorId());
            $tax_type = strtolower($settings['tax_type'] ?? 'gst');
            $company_state = strtolower($settings['company_state'] ?? '');
            $billing_state = strtolower($customer->billing_state ?? '');
            $isSameState = (!empty($company_state) && !empty($billing_state) && $company_state === $billing_state);
    
            // 🔥 Process items for tax calculations
            foreach ($iteams as $iteam) {
                $taxRate = $this->getTaxRate($iteam->tax);
                $iteam->totalTaxRate = $taxRate;
    
                $taxableAmount = ($iteam->price - $iteam->discount) * $iteam->quantity;
                $taxAmount = Utility::taxRate($taxRate, $iteam->price, $iteam->quantity, $iteam->discount);
    
                if ($tax_type === 'vat') {
                    // VAT only
                    $iteam->taxBreakdown = [
                        'VAT' => [
                            'rate' => $taxRate,
                            'amount' => $taxAmount,
                        ],
                    ];
                } else {
                    // GST logic
                    if ($isSameState) {
                        $halfRate = $taxRate / 2;
                        $halfAmount = $taxAmount / 2;
                        $iteam->taxBreakdown = [
                            'CGST' => [
                                'rate' => $halfRate,
                                'amount' => $halfAmount,
                            ],
                            'SGST' => [
                                'rate' => $halfRate,
                                'amount' => $halfAmount,
                            ],
                        ];
                    } else {
                        $iteam->taxBreakdown = [
                            'IGST' => [
                                'rate' => $taxRate,
                                'amount' => $taxAmount,
                            ],
                        ];
                    }
                }
    
                $iteam->totalWithTax = $taxableAmount + $taxAmount;
            }
    
            return view('proposal.view', compact(
                'proposal',
                'customer',
                'iteams',
                'status',
                'customFields',
                'tax_type',
                'company_state',
                'isSameState'
            ));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
    

    public function destroy(Proposal $proposal)
    {
        if (\Auth::user()->can('delete proposal')) {
            if ($proposal->created_by == \Auth::user()->creatorId()) {
                $proposal->delete();
                ProposalProduct::where('proposal_id', '=', $proposal->id)->delete();
                return redirect()->route('proposal.index')->with('success', __('Proposal successfully deleted.'));
            } else {
                return redirect()->back()->with('error', __('Permission denied.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function productDestroy(Request $request)
    {
        if (\Auth::user()->can('delete proposal product')) {
            ProposalProduct::where('id', '=', $request->id)->delete();
            return response()->json(['status' => true, 'message' => __('Proposal product successfully deleted.')]);
        } else {
            return response()->json(['status' => false, 'message' => __('Permission denied.')]);
        }
    }
    public function customerProposal(Request $request)
    {
        if (\Auth::user()->can('manage customer proposal')) {
            $status = Proposal::$statues;
            $query = Proposal::where('customer_id', '=', \Auth::user()->id)
                            ->where('status', '!=', '0')
                            ->where('created_by', \Auth::user()->creatorId());
    
            if (!empty($request->issue_date)) {
                $date_range = explode(' - ', $request->issue_date);
                $query->whereBetween('issue_date', $date_range);
            }
    
            if (!empty($request->status)) {
                $query->where('status', '=', $request->status);
            }
    
            $proposals = $query->get();
    
            return view('proposal.index', compact('proposals', 'status'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }
    public function customerProposalShow($ids)
    {
       
    
        if (!\Auth::user()->can('show proposal')) {
            \Log::warning('Permission denied: User lacks show proposal permission', [
                'user_id' => \Auth::user()->id,
            ]);
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    
        try {
            $proposal_id = Crypt::decrypt($ids);
            \Log::info('Decrypted proposal ID', [
                'encrypted_id' => $ids,
                'decrypted_id' => $proposal_id,
            ]);
    
            $proposal = Proposal::with(['items.product.unit', 'customer'])->find($proposal_id);
    
            if (!$proposal) {
                \Log::warning('Proposal not found', ['proposal_id' => $proposal_id]);
                return redirect()->back()->with('error', __('Proposal not found.'));
            }
    
            if ($proposal->created_by != \Auth::user()->creatorId()) {
                
                return redirect()->back()->with('error', __('Permission denied.'));
            }
    
            $customer = $proposal->customer;
            $items = $proposal->items ?? collect([]);
    
            $companySettings = Utility::settingsById($proposal->created_by);
    
            $companyState = strtolower(trim($companySettings['company_state'] ?? ''));
            $billingState = strtolower(trim($customer->billing_state ?? ''));
            $isSameState = (!empty($companyState) && !empty($billingState) && $companyState === $billingState);
    
           
    
            $customFields = CustomField::where('created_by', $proposal->created_by)
                ->where('module', 'proposal')
                ->get();
    
            return view('proposal.view', compact(
                'proposal',
                'customer',
                'items',
                'customFields',
               
                'companyState',
                'billingState',
                'isSameState'
            ));
        } catch (\Exception $e) {
            \Log::error('Proposal Show Error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return redirect()->back()->with('error', __('An error occurred while processing the proposal. Please try again.'));
        }
    }
    
    public function sent($id)
{
    if (!\Auth::user()->can('send proposal')) {
        return redirect()->back()->with('error', __('Permission denied.'));
    }

    $proposal = Proposal::find($id);
    if (!$proposal) {
        return redirect()->back()->with('error', __('Proposal not found.'));
    }

    $proposal->send_date = now();
    $proposal->status = 1;
    $proposal->save();

    $customer = Customer::find($proposal->customer_id);
    $proposal->name = $customer->name ?? '';
    $proposal->proposal = \Auth::user()->proposalNumberFormat($proposal->proposal_id);
    $proposal->url = route('proposal.pdf', Crypt::encrypt($proposal->id));

    $messages = [];

    // 📧 Email
    $settings = Utility::settings();
    if (!empty($settings['proposal_sent']) && $settings['proposal_sent'] == 1 && !empty($customer->email)) {
        $proposalArr = [
            'proposal_name' => $proposal->name,
            'proposal_number' => $proposal->proposal,
            'proposal_url' => $proposal->url,
        ];
        $resp = Utility::sendEmailTemplate('proposal_sent', [$customer->id => $customer->email], $proposalArr);
        if (!$resp['is_success'] && !empty($resp['error'])) {
            $messages[] = '<span class="text-danger">Email Error: ' . $resp['error'] . '</span>';
        } else {
            $messages[] = 'Email sent ✅';
        }
    }

    // 📲 WhatsApp
    if (!empty($customer->contact)) {
        $number = preg_replace('/[^0-9]/', '', $customer->contact); // Sanitize number
        $message = "Dear {$customer->name}, your proposal ({$proposal->proposal}) is ready. Please check your email.";
        $whatsappStatus = Utility::sendWhatsappMessage($number, $message);

        \Log::info("📤 WhatsApp to {$number} status: " . ($whatsappStatus ? 'Success' : 'Failed'));
        $messages[] = $whatsappStatus ? 'WhatsApp sent ✅' : '<span class="text-danger">WhatsApp failed ❌</span>';
    } else {
        \Log::warning("⚠️ No contact for customer ID {$customer->id}.");
        $messages[] = '<span class="text-warning">No contact number found.</span>';
    }

    return redirect()->back()->with('success', __('Proposal successfully sent.') . '<br>' . implode('<br>', $messages));
}


   public function resent($id)
{
    if (!\Auth::user()->can('send proposal')) {
        return redirect()->back()->with('error', __('Permission denied.'));
    }

    $proposal = Proposal::find($id);
    if (!$proposal) {
        return redirect()->back()->with('error', __('Proposal not found.'));
    }

    $customer = Customer::find($proposal->customer_id);
    $proposal->name = $customer->name ?? '';
    $proposal->proposal = \Auth::user()->proposalNumberFormat($proposal->proposal_id);
    $proposal->url = route('proposal.pdf', Crypt::encrypt($proposal->id));

    $messages = [];

    // 📧 Email
    $settings = Utility::settings();
    if (!empty($settings['proposal_sent']) && !empty($customer->email)) {
        $proposalArr = [
            'proposal_name' => $proposal->name,
            'proposal_number' => $proposal->proposal,
            'proposal_url' => $proposal->url,
        ];
        $resp = Utility::sendEmailTemplate('proposal_sent', [$customer->id => $customer->email], $proposalArr);  
    }

    
    if (!empty($customer->contact)) {
        $number = preg_replace('/[^0-9]/', '', $customer->contact);
        $message = "Dear {$customer->name}, your proposal ({$proposal->proposal}) is ready. Please check your email.";
        $whatsappStatus = Utility::sendWhatsappMessage($number, $message);

    } 

    return redirect()->back()->with('success', __('Proposal successfully resent.') . '<br>' . implode('<br>', $messages));
}

    public function shippingDisplay(Request $request, $id)
    {
        $proposal = Proposal::find($id);
        $proposal->shipping_display = $request->is_display == 'true' ? 1 : 0;
        $proposal->save();
        return redirect()->back()->with('success', __('Shipping address status successfully changed.'));
    }

    public function duplicate($proposal_id)
    {
        if (\Auth::user()->can('duplicate proposal')) {
            $proposal = Proposal::where('id', $proposal_id)->first();
            $duplicateProposal = new Proposal();
            $duplicateProposal->proposal_id = $this->proposalNumber();
            $duplicateProposal->customer_id = $proposal['customer_id'];
            $duplicateProposal->issue_date = date('Y-m-d');
            $duplicateProposal->send_date = null;
            $duplicateProposal->category_id = $proposal['category_id'];
            $duplicateProposal->status = 0;
            $duplicateProposal->created_by = $proposal['created_by'];
            $duplicateProposal->save();

            if ($duplicateProposal) {
                $proposalProduct = ProposalProduct::where('proposal_id', $proposal_id)->get();
                foreach ($proposalProduct as $product) {
                    $duplicateProduct = new ProposalProduct();
                    $duplicateProduct->proposal_id = $duplicateProposal->id;
                    $duplicateProduct->product_id = $product->product_id;
                    $duplicateProduct->quantity = $product->quantity;
                    $duplicateProduct->tax = $product->tax;
                    $duplicateProduct->discount = $product->discount;
                    $duplicateProduct->price = $product->price;
                    $duplicateProduct->save();
                }
            }

            return redirect()->back()->with('success', __('Proposal duplicate successfully.'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function convert($proposal_id)
    {
        if (\Auth::user()->can('convert invoice')) {
            $proposal = Proposal::where('id', $proposal_id)->first();
            $proposal->is_convert = 1;
            $proposal->save();

            $convertInvoice = new Invoice();
            $convertInvoice->invoice_id = $this->invoiceNumber();
            $convertInvoice->customer_id = $proposal['customer_id'];
            $convertInvoice->issue_date = date('Y-m-d');
            $convertInvoice->due_date = date('Y-m-d');
            $convertInvoice->send_date = null;
            $convertInvoice->category_id = $proposal['category_id'];
            $convertInvoice->status = 0;
            $convertInvoice->created_by = $proposal['created_by'];
            $convertInvoice->save();

            $proposal->converted_invoice_id = $convertInvoice->id;
            $proposal->save();

            if ($convertInvoice) {
                $proposalProduct = ProposalProduct::where('proposal_id', $proposal_id)->get();
                foreach ($proposalProduct as $product) {
                    $duplicateProduct = new InvoiceProduct();
                    $duplicateProduct->invoice_id = $convertInvoice->id;
                    $duplicateProduct->product_id = $product->product_id;
                    $duplicateProduct->quantity = $product->quantity;
                    $duplicateProduct->tax = $product->tax;
                    $duplicateProduct->discount = $product->discount;
                    $duplicateProduct->price = $product->price;
                    $duplicateProduct->save();

                    Utility::total_quantity('minus', $duplicateProduct->quantity, $duplicateProduct->product_id);

                    $type = 'invoice';
                    $type_id = $convertInvoice->id;
                    StockReport::where('type', '=', 'invoice')->where('type_id', '=', $convertInvoice->id)->delete();
                    $description = $duplicateProduct->quantity . __(' quantity sold in') . ' ' . \Auth::user()->proposalNumberFormat($proposal->proposal_id) . ' ' . __('Proposal convert to invoice') . ' ' . \Auth::user()->invoiceNumberFormat($convertInvoice->invoice_id);
                    Utility::addProductStock($duplicateProduct->product_id, $duplicateProduct->quantity, $type, $description, $type_id);
                }
            }

            return redirect()->back()->with('success', __('Proposal to invoice convert successfully.'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function statusChange(Request $request, $id)
    {
        $status = $request->status;
        $proposal = Proposal::find($id);
        $proposal->status = $status;
        $proposal->save();
        return redirect()->back()->with('success', __('Proposal status changed successfully.'));
    }

    public function previewProposal($template, $color)
    {
       
        $objUser = \Auth::user();
        $settings = Utility::settings();
        $proposal = new Proposal();

        $customer = new \stdClass();
        $customer->email = '<Email>';
        $customer->shipping_name = '<Customer Name>';
        $customer->shipping_country = '<Country>';
        $customer->shipping_state = '<State>';
        $customer->shipping_city = '<City>';
        $customer->shipping_phone = '<Customer Phone Number>';
        $customer->shipping_zip = '<Zip>';
        $customer->shipping_address = '<Address>';
        $customer->billing_name = '<Customer Name>';
        $customer->billing_country = '<Country>';
        $customer->billing_state = '<State>';
        $customer->billing_city = '<City>';
        $customer->billing_phone = '<Customer Phone Number>';
        $customer->billing_zip = '<Zip>';
        $customer->billing_address = '<Address>';

        $totalTaxPrice = 0;
        $taxesData = [];

        $items = [];
        for ($i = 1; $i <= 3; $i++) {
            $item = new \stdClass();
            $item->name = 'Item ' . $i;
            $item->quantity = 1;
            $item->tax = 5;
            $item->discount = 50;
            $item->price = 100;
            $item->unit = 1;

            $taxes = ['Tax 1', 'Tax 2'];
            $itemTaxes = [];
            foreach ($taxes as $k => $tax) {
                $taxPrice = 10;
                $totalTaxPrice += $taxPrice;
                $itemTax['name'] = 'Tax ' . $k;
                $itemTax['rate'] = '10 %';
                $itemTax['price'] = '$10';
                $itemTax['tax_price'] = 10;
                $itemTaxes[] = $itemTax;
                $taxesData['Tax ' . $k] = array_key_exists('Tax ' . $k, $taxesData) ? $taxesData['Tax ' . $k] + $taxPrice : $taxPrice;
            }
            $item->itemTax = $itemTaxes;
            $items[] = $item;
        }

        $proposal->proposal_id = 1;
        $proposal->issue_date = date('Y-m-d H:i:s');
        $proposal->due_date = date('Y-m-d H:i:s');
        $proposal->itemData = $items;
        $proposal->totalTaxPrice = 60;
        $proposal->totalQuantity = 3;
        $proposal->totalRate = 300;
        $proposal->totalDiscount = 10;
        $proposal->taxesData = $taxesData;
        $proposal->created_by = $objUser->creatorId();
        $accounts = BankAccount::where('created_by', $objUser->creatorId())->get();
        $proposal->customField = [];
        $customFields = [];

        $preview = 1;
        $color = '#' . $color;
        $fontColor = Utility::getFontColor($color);

    $company_logo = Utility::getValByName('company_logo_dark');
$proposal_logo = Utility::getValByName('proposal_logo');

$img = isset($proposal_logo) && !empty($proposal_logo)
    ? asset('storage/proposal_logo/' . $proposal_logo)
    : asset('storage/uploads/logo/' . (!empty($company_logo) ? $company_logo : 'logo-dark.png'));



        return view('proposal.templates.' . $template, compact('proposal', 'preview', 'color', 'img', 'settings', 'customer', 'fontColor', 'customFields', 'accounts'));
    }
public function saveProposalTemplateSettings(Request $request)
{
    $post = $request->all();
    unset($post['_token']);

    if (isset($post['proposal_template']) && (!isset($post['proposal_color']) || empty($post['proposal_color']))) {
        $post['proposal_color'] = "ffffff";
    }

    if ($request->hasFile('proposal_logo')) {
        $dir = 'proposal_logo/';
        $proposal_logo = \Auth::user()->id . '_proposal_logo.png';
        $validation = [
            'mimes:png',
            'max:20480', 
        ];
        $path = Utility::upload_file($request, 'proposal_logo', $proposal_logo, $dir, $validation);

        if ($path['flag'] == 0) {
            return redirect()->back()->with('error', __($path['msg']));
        }
        $post['proposal_logo'] = $proposal_logo;
    }

    foreach ($post as $key => $data) {
        \DB::insert(
            'insert into settings (`value`, `name`, `created_by`) values (?, ?, ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)',
            [$data, $key, \Auth::user()->creatorId()]
        );
    }

    return redirect()->back()->with('success', __('Proposal Setting updated successfully'));
}

    function invoiceNumber()
    {
        $latest = Invoice::where('created_by', '=', \Auth::user()->creatorId())->latest()->first();
        return $latest ? $latest->invoice_id + 1 : 1;
    }

    public function items(Request $request)
    {
        $items = ProposalProduct::where('proposal_id', $request->proposal_id)->where('product_id', $request->product_id)->first();
        return json_encode($items);
    }

    public function invoiceLink($proposalID)
    {
        try {
            $id = Crypt::decrypt($proposalID);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', __('Proposal Not Found.'));
        }

        $proposal = Proposal::find($id);
        if (!empty($proposal)) {
            $user_id = $proposal->created_by;
            $user = User::find($user_id);
            $customer = $proposal->customer;
            $iteams = $proposal->items;
            $proposal->customField = CustomField::getData($proposal, 'proposal');
            $status = Proposal::$statues;
            $customFields = CustomField::where('module', '=', 'proposal')->get();

            return view('proposal.customer_proposal', compact('proposal', 'customer', 'iteams', 'customFields', 'status', 'user'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied.'));
        }
    }

    public function export()
    {
        $name = 'proposal_' . date('Y-m-d i:h:s');
        $data = Excel::download(new ProposalExport(), $name . '.xlsx');
        ob_end_clean();
        return $data;
    }
}