<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\Utility; // For getTaxData()
use App\Models\InvoiceProduct;
use App\Models\InvoicePayment;
use App\Models\InvoiceBankTransfer;
use App\Models\Customer;
use App\Models\CreditNote;
use App\Models\ProductServiceCategory;

class Invoice extends Model
{
    protected $fillable = [
        'invoice_id',
        'customer_id',
        'issue_date',
        'due_date',
        'ref_number',
        'status',
        'is_convert',
        'converted_ewayBill_id',
        'category_id',
        'created_by',
    ];

    public static $statues = [
        'Draft',
        'Sent',
        'Unpaid',
        'Partialy Paid',
        'Paid',
    ];

    // Relationships
    public function items()
    {
        return $this->hasMany(InvoiceProduct::class, 'invoice_id', 'id');
    }

    public function payments()
    {
        return $this->hasMany(InvoicePayment::class, 'invoice_id', 'id');
    }

    public function bankPayments()
    {
        return $this->hasMany(InvoiceBankTransfer::class, 'invoice_id', 'id')->where('status', '!=', 'Approved');
    }

    public function customer()
    {
        return $this->hasOne(Customer::class, 'id', 'customer_id');
    }

    public function category()
    {
        return $this->hasOne(ProductServiceCategory::class, 'id', 'category_id');
    }

    public function creditNote()
    {
        return $this->hasMany(CreditNote::class, 'invoice', 'id');
    }

    public function lastPayments()
    {
        return $this->hasOne(InvoicePayment::class, 'invoice_id', 'id')->latest('created_at');
    }

 public function getTotal()
{
    $subTotal = $this->getSubTotal();
    $discount = $this->getTotalDiscount();
    $tax = $this->getTotalTax();

    $rawTotal = ($subTotal - $discount) + $tax;
    $roundedTotal = round($rawTotal);

    return $roundedTotal;
}

    public function getSubTotal()
    {
        $subTotal = 0;
        foreach ($this->items as $product) {
            $subTotal += ($product->price * $product->quantity);
        }
        return $subTotal;
    }

 public function getTotalTax()
{
    $totalTax = 0;

    foreach ($this->items as $product) {
        $discount = isset($product->discount) ? $product->discount : 0;
        $quantity = $product->quantity ?? 1;
        $price = $product->price ?? 0;

        $taxableAmount = ($price - $discount) * $quantity;

        // If tax is stored as JSON or comma-separated string
        $taxIds = [];

        if (!empty($product->tax)) {
            if (is_array($product->tax)) {
                $taxIds = $product->tax;
            } elseif (is_string($product->tax)) {
                // Try JSON decode first
                $decoded = json_decode($product->tax, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $taxIds = is_array($decoded) ? $decoded : [];
                } else {
                    $taxIds = explode(',', $product->tax);
                }
            }
        }

        $totalRate = 0;
        foreach ($taxIds as $taxId) {
            $taxId = trim($taxId);
            $tax = \App\Models\Tax::find($taxId);
            if ($tax) {
                $totalRate += (float) $tax->rate;
            }
        }

        $totalTax += ($taxableAmount * $totalRate) / 100;
    }

    return $totalTax;
}



    public function getTotalDiscount()
    {
        $totalDiscount = 0;
        foreach ($this->items as $product) {
            $totalDiscount += $product->discount;
        }
        return $totalDiscount;
    }

public function getDue()
{
    $paid = 0;
    foreach ($this->payments as $payment) {
        $paid += $payment->amount;
    }
    $creditNotes = $this->invoiceTotalCreditNote();
    $due = $this->getTotal() - $paid - $creditNotes;
    return max(0, $due); // Ensure due amount is never negative
}

    // Utility Methods
    public static function change_status($invoice_id, $status)
    {
        $invoice = Invoice::find($invoice_id);
        if ($invoice) {
            $invoice->status = $status;
            $invoice->update();
        }
    }

    public function invoiceTotalCreditNote()
{
    $total = CreditNote::where('invoice', $this->id)->sum('amount');
    return max(0, $total); // Ensure credit note total is non-negative
}
}