<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\Utility; // For getTaxData()
use App\Models\InvoiceProduct;
use App\Models\InvoicePayment;
use App\Models\InvoiceBankTransfer;
use App\Models\Customer;
use App\Models\CreditNote;
use App\Models\ProductServiceCategory;
use App\Models\Invoice;

class EWayBill extends Model
{    
    protected $table = 'ewaybills';
    protected $fillable = [
        'ewaybill_id',
        'customer_id',
        'e_way_bill_number',
        'delivery_distance',
        'issue_date',
        'due_date',
        'ref_number',
        'status',
        'category_id',
        'created_by',
    ];

    public static $statues = [
        'Draft',
        'Sent',
        'Unpaid',
        'Partialy Paid',
        'Paid',
    ];

    // Relationships
    public function items()
    {
        return $this->hasMany(EWayBillProduct::class, 'ewaybill_id', 'id');
    }

    public function payments()
    {
        return $this->hasMany(InvoicePayment::class, 'ewaybill_id', 'id');
    }

    public function bankPayments()
    {
        return $this->hasMany(InvoiceBankTransfer::class, 'ewaybill_id', 'id')->where('status', '!=', 'Approved');
    }

    public function customer()
    {
        return $this->hasOne(Customer::class, 'id', 'customer_id');
    }

    public function category()
    {
        return $this->hasOne(ProductServiceCategory::class, 'id', 'category_id');
    }

    public function creditNote()
    {
        return $this->hasMany(CreditNote::class, 'ewaybill', 'id');
    }

    public function lastPayments()
    {
        return $this->hasOne(InvoicePayment::class, 'ewaybill_id', 'id')->latest('created_at');
    }

 public function getTotal()
{
    $subTotal = $this->getSubTotal();
    $discount = $this->getTotalDiscount();
    $tax = $this->getTotalTax();

    $rawTotal = ($subTotal - $discount) + $tax;
    $roundedTotal = round($rawTotal);

    return $roundedTotal;
}
    public function getSubTotal()
    {
        $subTotal = 0;
        foreach ($this->items as $product) {
            $subTotal += ($product->price * $product->quantity);
        }
        return $subTotal;
    }

 public function getTotalTax()
{
    $totalTax = 0;

    foreach ($this->items as $product) {
        $discount = isset($product->discount) ? $product->discount : 0;
        $quantity = $product->quantity ?? 1;
        $price = $product->price ?? 0;

        $taxableAmount = ($price - $discount) * $quantity;

        // If tax is stored as JSON or comma-separated string
        $taxIds = [];

        if (!empty($product->tax)) {
            if (is_array($product->tax)) {
                $taxIds = $product->tax;
            } elseif (is_string($product->tax)) {
                // Try JSON decode first
                $decoded = json_decode($product->tax, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $taxIds = is_array($decoded) ? $decoded : [];
                } else {
                    $taxIds = explode(',', $product->tax);
                }
            }
        }

 $totalRate = 0;

if (!empty($product->tax)) {
    // Ensure tax is stored as comma-separated IDs like "1,2,3"
    $taxIds = explode(',', $product->tax);

    // Sanitize and cast to integers
    $taxIds = array_map('intval', array_filter($taxIds));

    // Get tax records from DB
    $taxes = \App\Models\Tax::whereIn('id', $taxIds)->get();

    // Sum valid tax rates
    foreach ($taxes as $tax) {
        if (is_numeric($tax->rate)) {
            $totalRate += (float) $tax->rate;
        }
    }
}


        $totalTax += ($taxableAmount * $totalRate) / 100;
    }

    return $totalTax;
}


    public function getTotalDiscount()
    {
        $totalDiscount = 0;
        foreach ($this->items as $product) {
            $totalDiscount += $product->discount;
        }
        return $totalDiscount;
    }

    public function getDue()
    {
        $due = 0;
        foreach ($this->payments as $payment) {
            $due += $payment->amount;
        }
        return ($this->getTotal() - $due) - $this->ewaybillTotalCreditNote();
    }

    // Utility Methods
    public static function change_status($ewaybill_id, $status)
    {
        $ewaybill = EWayBill::find($ewaybill_id);
        if ($ewaybill) {
            $ewaybill->status = $status;
            $ewaybill->update();
        }
    }

    public function ewaybillTotalCreditNote()
    {
        return $this->creditNote->sum('amount') ?? 0;
    }
}