<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\Leave;
use App\Models\LeaveType;
use App\Models\Utility;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class LeaveController extends Controller
{
   public function index()
    {
        if (!Auth::user()->can('manage leave')) {
          
            return redirect()->back()->with('error', __('Permission denied.'));
        }

        $leaves = collect();
        $user = Auth::user();

        if ($user->type == 'company' || $user->type == 'HR') {
            $leaves = Leave::where('created_by', $user->creatorId())
                ->with(['leaveType', 'employees'])
                ->orderByDesc('id')
                ->get();
         } elseif ($user->type == 'Team Lead') {
            $tl_employee = Employee::where('user_id', $user->id)->first();
            if (!$tl_employee) {
             
                return redirect()->back()->with('error', __('No employee record found for this Team Lead. Please contact HR to set up your employee profile.'));
            }
            $employee_ids = Employee::where('team_lead', 'Team Lead (' . $tl_employee->email . ')')
                ->pluck('id');
            if ($employee_ids->isEmpty()) {
                
                return view('leave.index', compact('leaves'))->with('warning', __('No employees are assigned to you as a Team Lead.'));
            }
            $leaves = Leave::whereIn('employee_id', $employee_ids)
                ->whereIn('status', ['Pending', 'tl_approved', 'Approved', 'Reject']) // Include relevant statuses
                ->with(['leaveType' => function ($query) {
                    $query->whereIn('title', ['Sick Leave', 'Casual Leave']);
                }, 'employees'])
                ->orderByDesc('id')
                ->get();
            if ($leaves->isEmpty()) {
                
                return view('leave.index', compact('leaves'))->with('info', __('No relevant leave requests found for your team.'));
            }
        } elseif ($user->type == 'Reporting Manager') {
            $manager_employee = Employee::where('user_id', $user->id)->first();
            if (!$manager_employee) {
              
                return redirect()->back()->with('error', __('No employee record found for this Reporting Manager.'));
            }
            $employee_ids = Employee::where('reporting_manager', 'Reporting Manager (' . $manager_employee->email . ')')
                ->pluck('id');
            if ($employee_ids->isEmpty()) {
               
                return view('leave.index', compact('leaves'))->with('warning', __('No employees are assigned to you as a Reporting Manager.'));
            }
            $leaves = Leave::whereIn('employee_id', $employee_ids)
                ->whereIn('status', ['tl_approved', 'Approved', 'Reject']) // Include tl_approved, Approved, and Reject for Reporting Manager
                ->with(['leaveType' => function ($query) {
                    $query->whereIn('title', ['Sick Leave', 'Casual Leave']);
                }, 'employees'])
                ->orderByDesc('id')
                ->get();
            if ($leaves->isEmpty()) {
               
                return view('leave.index', compact('leaves'))->with('info', __('No TL Approved, Approved, or Rejected Sick Leave or Casual Leave found for your team.'));
            }
        } else {
            $employee = Employee::where('user_id', $user->id)->first();
            if ($employee) {
                $leaves = Leave::where('employee_id', $employee->id)
                    ->with(['leaveType' => function ($query) {
                        $query->whereIn('title', ['Sick Leave', 'Casual Leave']);
                    }, 'employees'])
                    ->orderByDesc('id')
                    ->get();
            } else {
              
                return redirect()->back()->with('error', __('No employee record found for this user.'));
            }
        }

        return view('leave.index', compact('leaves'));
    }


    public function create()
    {
        if (!Auth::user()->can('create leave')) {
            return response()->json(['error' => __('Permission denied.')], 401);
        }

        $employee_id = null;
        $user = Auth::user();
        $employees = collect();

        if ($user->type == 'company' || $user->type == 'HR') {
            $employees = Employee::where('created_by', $user->creatorId())->pluck('name', 'id');
        } else {
            $employee = Employee::where('user_id', $user->id)->first();
            if ($employee) {
                $employees = [$employee->id => $employee->name];
                $employee_id = $employee->id;
            }
        }

        $leavetypes = LeaveType::where('created_by', $user->creatorId())->get();

        return view('leave.create', compact('employees', 'leavetypes', 'employee_id'));
    }

    public function store(Request $request)
    {
        if (!Auth::user()->can('create leave')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }

        $validator = Validator::make($request->all(), [
            'leave_type_id' => 'required|exists:leave_types,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'leave_reason' => 'required|string',
            'remark' => 'required|string',
            'employee_id' => 'required_if:type,company,HR|exists:employees,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->with('error', $validator->errors()->first());
        }

        $user = Auth::user();
        $employee_id = ($user->type == 'company' || $user->type == 'HR') ? $request->employee_id : Employee::where('user_id', $user->id)->first()->id;

        // Check for overlapping leave requests
        $existing_leave = Leave::where('employee_id', $employee_id)
            ->where('status', '!=', 'Reject')
            ->where(function ($query) use ($request) {
                $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                      ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                      ->orWhere(function ($q) use ($request) {
                          $q->where('start_date', '<=', $request->start_date)
                            ->where('end_date', '>=', $request->end_date);
                      });
            })
            ->exists();

            if ($existing_leave) {
                return redirect()->back()->with('error', __('A leave request already exists for the selected dates.'));
            } else {
                $rejected_leave = Leave::where('employee_id', $employee_id)
                    ->where('status', 'Reject')
                    ->where(function ($query) use ($request) {
                        $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                              ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                              ->orWhere(function ($q) use ($request) {
                                  $q->where('start_date', '<=', $request->start_date)
                                    ->where('end_date', '>=', $request->end_date);
                              });
                    })
                    ->exists();
                if ($rejected_leave) {
                    \Log::info('New leave application for previously rejected dates: Employee ID ' . $employee_id . ', Dates ' . $request->start_date . ' to ' . $request->end_date);
                }
            }

        $leave_type = LeaveType::find($request->leave_type_id);
        $startDate = new \DateTime($request->start_date);
        $endDate = new \DateTime($request->end_date);
        $endDate->add(new \DateInterval('P1D'));
        $total_leave_days = $startDate->diff($endDate)->days;

        if ($leave_type->days < $total_leave_days) {
            return redirect()->back()->with('error', __('Leave type ' . $leave_type->title . ' allows maximum ' . $leave_type->days . ' days.'));
        }

        $leave = new Leave();
        $leave->employee_id = $employee_id;
        $leave->leave_type_id = $request->leave_type_id;
        $leave->applied_on = now()->format('Y-m-d');
        $leave->start_date = $request->start_date;
        $leave->end_date = $request->end_date;
        $leave->total_leave_days = $total_leave_days;
        $leave->leave_reason = $request->leave_reason;
        $leave->remark = $request->remark;
        $leave->status = ($user->type == 'company') ? 'Approved' : 'Pending';
        $leave->created_by = $user->creatorId();
        $leave->save();

      
if ($user->type != 'company' && $user->type != 'HR') {
    $settings = Utility::settings($user->creatorId());
    $employee = Employee::find($leave->employee_id);

  
    if ($employee && isset($settings['new_leave']) && $settings['new_leave'] == 1) {
        $teamLeadRaw = $employee->team_lead;

        // Extract email from string like "Team Lead (tl@example.com)"
        if (preg_match('/\(([^)]+)\)/', $teamLeadRaw, $matches)) {
            $teamLeadEmail = $matches[1];
        } else {
            $teamLeadEmail = $teamLeadRaw; // fallback if already plain email
        }

        $tlUser = User::where('email', $teamLeadEmail)->first();

        if ($tlUser) {
           
            $tlEmployee = Employee::where('user_id', $tlUser->id)->first();

            $leaveArr = [
                'user_name'     => $user->name,
                'start_date'    => $leave->start_date,
                'end_date'      => $leave->end_date,
                'leave_reason'  => $leave->leave_reason,
                'employee_name' => $employee->name,
            ];

            // Email
            Utility::sendEmailTemplate('new_leave', [$tlUser->id => $teamLeadEmail], $leaveArr);
       
            // WhatsApp
            if (!empty($tlEmployee) && !empty($tlEmployee->phone)) {
                $phone = preg_replace('/[^0-9]/', '', $tlEmployee->phone);
                $message = "Hello {$tlEmployee->name},\n"
                    . "You have a new leave request from *{$employee->name}*.\n"
                    . "From: {$leave->start_date} to {$leave->end_date}\n"
                    . "Reason: {$leave->leave_reason}";

                Utility::sendWhatsappMessage($phone, $message, $user->creatorId());
                
            }
        } 
    }
}
         elseif ($user->type == 'company' && isset($settings['leave_action_sent']) && $settings['leave_action_sent'] == 1) {
            // Notify employee if company directly approves
            $employee = Employee::find($leave->employee_id);
            $leaveArr = [
                'leave_name' => $employee->name,
                'leave_status' => 'Approved',
                'leave_reason' => $leave->leave_reason,
                'leave_start_date' => $leave->start_date,
                'leave_end_date' => $leave->end_date,
                'total_leave_days' => $leave->total_leave_days,
            ];
            $resp = Utility::sendEmailTemplate('leave_action_sent', [$employee->user_id => $employee->email], $leaveArr);
            if (!$resp['is_success']) {
                \Log::error('Email Sending Failed for Leave Action: Leave ID ' . $leave->id . ', Error ' . ($resp['error'] ?? 'Unknown error') . ', User ID ' . $user->id);
            }
        }

        return redirect()->route('leave.index')->with('success', __('Leave successfully created.'));
    }

    public function show(Leave $leave)
    {
        if (!Auth::user()->can('manage leave')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
        return view('leave.show', compact('leave'));
    }

    public function edit(Leave $leave)
    {
        if (!Auth::user()->can('edit leave')) {
            return response()->json(['error' => __('Permission denied.')], 401);
        }

        if ($leave->created_by != Auth::user()->creatorId()) {
            return response()->json(['error' => __('Permission denied.')], 401);
        }

        $user = Auth::user();
        $employees = collect();
        $employee_id = null;

        if ($user->type == 'company' || $user->type == 'HR') {
            $employees = Employee::where('created_by', $user->creatorId())->pluck('name', 'id');
        } else {
            $employee = Employee::where('user_id', $user->id)->first();
            if ($employee) {
                $employees = [$employee->id => $employee->name];
                $employee_id = $employee->id;
            }
        }

        $leavetypes = LeaveType::where('created_by', $user->creatorId())->pluck('title', 'id');

        return view('leave.edit', compact('leave', 'employees', 'leavetypes', 'employee_id'));
    }

    public function update(Request $request, $leave)
    {
        $leave = Leave::findOrFail($leave);
        $user = Auth::user();
    
        if (!$user->can('edit leave')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    
        if ($leave->created_by != $user->creatorId()) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    
        if ($user->type == 'Employee' && $leave->status != 'Pending' && $leave->status != 'Reject') {
            return redirect()->back()->with('error', __('You can only edit leaves with a Pending or Rejected status.'));
        }
    
        if ($user->type == 'Team Lead' || $user->type == 'Reporting Manager') {
            return redirect()->back()->with('error', __('Please use the approval actions to manage leaves.'));
        }
    
        $validator = Validator::make($request->all(), [
            'leave_type_id' => 'required|exists:leave_types,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'leave_reason' => 'required|string',
            'remark' => 'required|string',
            'employee_id' => 'required_if:type,company,HR|exists:employees,id',
        ]);
    
        if ($validator->fails()) {
            return redirect()->back()->with('error', $validator->errors()->first());
        }
    
        $employee_id = ($user->type == 'company' || $user->type == 'HR') ? $request->employee_id : Employee::where('user_id', $user->id)->first()->id;
    
        // Check for overlapping leave requests, excluding the current leave
        $existing_leave = Leave::where('employee_id', $employee_id)
            ->where('id', '!=', $leave->id)
            ->where('status', '!=', 'Reject')
            ->where(function ($query) use ($request) {
                $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                      ->orWhereBetween('end_date', [$request->start_date, $request->end_date])
                      ->orWhere(function ($q) use ($request) {
                          $q->where('start_date', '<=', $request->start_date)
                            ->where('end_date', '>=', $request->end_date);
                      });
            })
            ->exists();
    
        if ($existing_leave) {
            return redirect()->back()->with('error', __('A leave request already exists for the selected dates.'));
        }
    
        $leave_type = LeaveType::find($request->leave_type_id);
        $startDate = new \DateTime($request->start_date);
        $endDate = new \DateTime($request->end_date);
        $endDate->add(new \DateInterval('P1D'));
        $total_leave_days = $startDate->diff($endDate)->days;
    
        if ($leave_type->days < $total_leave_days) {
            return redirect()->back()->with('error', __('Leave type ' . $leave_type->title . ' allows maximum ' . $leave_type->days . ' days.'));
        }
    
        if ($user->type == 'company' && $request->status) {
            $leave->status = $request->status;
        } elseif ($user->type == 'HR' && $request->status) {
            if ($request->status == 'Approved' && $leave->status != 'tl_approved') {
                return redirect()->back()->with('error', __('Cannot set status to Approved unless it is TL Approved.'));
            }
            if ($request->status == 'tl_approved' && $leave->status != 'Pending') {
                return redirect()->back()->with('error', __('Cannot set status to TL Approved unless it is Pending.'));
            }
            $leave->status = $request->status;
        } else {
            // Reset status to Pending for Employee editing a Rejected or Pending leave
            $leave->status = 'Pending';
        }
    
        $leave->employee_id = $request->employee_id ?? $leave->employee_id;
        $leave->leave_type_id = $request->leave_type_id;
        $leave->start_date = $request->start_date;
        $leave->end_date = $request->end_date;
        $leave->total_leave_days = $total_leave_days;
        $leave->leave_reason = $request->leave_reason;
        $leave->remark = $request->remark;
    
        $leave->save();
    
        // Re-notify Team Leader if Employee edits
        if ($user->type == 'Employee') {
            $settings = Utility::settings($user->creatorId());
            $employee = Employee::find($leave->employee_id);
            if ($employee->team_leader_id && isset($settings['new_leave']) && $settings['new_leave'] == 1) {
                $team_leader = Employee::find($employee->team_leader_id);
                $tl_user = User::find($team_leader->user_id);
                if ($tl_user) {
                    $leaveArr = [
                        'user_name' => $user->name,
                        'start_date' => $leave->start_date,
                        'end_date' => $leave->end_date,
                        'leave_reason' => $leave->leave_reason,
                        'employee_name' => $employee->name,
                    ];
                    Utility::sendEmailTemplate('new_leave', [$tl_user->id => $tl_user->email], $leaveArr);
                }
            }
        } 

        elseif ($user->type == 'company' && $leave->status == 'Approved' && isset($settings['leave_action_sent']) && $settings['leave_action_sent'] == 1) {
            // Notify employee if company approves during update
            $employee = Employee::find($leave->employee_id);
            $leaveArr = [
                'leave_name' => $employee->name,
                'leave_status' => 'Approved',
                'leave_reason' => $leave->leave_reason,
                'leave_start_date' => $leave->start_date,
                'leave_end_date' => $leave->end_date,
                'total_leave_days' => $leave->total_leave_days,
            ];
            $resp = Utility::sendEmailTemplate('leave_action_sent', [$employee->user_id => $employee->email], $leaveArr);
            if (!$resp['is_success']) {
                \Log::error('Email Sending Failed for Leave Action: Leave ID ' . $leave->id . ', Error ' . ($resp['error'] ?? 'Unknown error') . ', User ID ' . $user->id);
            }
        }
    
        return redirect()->route('leave.index')->with('success', __('Leave successfully updated.'));
    }
    public function destroy(Leave $leave)
    {
        if (!Auth::user()->can('delete leave')) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }

        if ($leave->created_by != Auth::user()->creatorId()) {
            return redirect()->back()->with('error', __('Permission denied.'));
        }

        $leave->delete();
        return redirect()->route('leave.index')->with('success', __('Leave successfully deleted.'));
    }

    public function action($id)
    {
        $leave = Leave::findOrFail($id);
        $employee = Employee::findOrFail($leave->employee_id);
        $leavetype = LeaveType::findOrFail($leave->leave_type_id);
        $type = Auth::user()->type;

        return view('leave.action', compact('employee', 'leavetype', 'leave', 'type'));
    }

    public function changeaction(Request $request)
    {
       
        $validator = Validator::make($request->all(), [
            'leave_id' => 'required|exists:leaves,id',
            'status' => 'required|in:tl_approved,Approved,Reject',
        ]);

        if ($validator->fails()) {
           
            return redirect()->route('leave.index')->with('error', $validator->errors()->first());
        }

        $leave = Leave::findOrFail($request->leave_id);
        $user = Auth::user();
        $employee = Employee::findOrFail($leave->employee_id);

//         dd([
//     'user_type' => $user->type,
//     'leave_status' => $leave->status,
// ]);
        if ($user->type == 'Team Lead' && $leave->status != 'Pending') {
          
            return redirect()->route('leave.index')->with('error', __('Only Pending leaves can be actioned by Team Lead.'));
        }

        if ($user->type == 'Reporting Manager' && $leave->status != 'tl_approved') {
           
            return redirect()->route('leave.index')->with('error', __('Only TL Approved leaves can be actioned by Reporting Manager.'));
        }

        $manager_employee = Employee::where('user_id', $user->id)->first();
        if (!$manager_employee) {
           
            return redirect()->route('leave.index')->with('error', __('No employee record found for this user.'));
        }

        // if ($user->type == 'Team Lead' && $employee->team_lead != 'Team Lead (' . $manager_employee->email . ')') {
        if ($user->type == 'Team Lead' && strcasecmp($employee->team_lead, 'Team Lead (' . $manager_employee->email . ')') !== 0) {
          
            return redirect()->route('leave.index')->with('error', __('You are not the Team Lead for this employee.'));
        } elseif ($user->type == 'Reporting Manager' && strcasecmp($employee->reporting_manager, 'Reporting Manager (' . $manager_employee->email . ')') !== 0) {
          
            return redirect()->route('leave.index')->with('error', __('You are not the Reporting Manager for this employee.'));
        }

        if ($user->type == 'company') {
            $leave->status = $request->status;
        } elseif ($user->type == 'Team Lead') {
            if ($request->status != 'tl_approved' && $request->status != 'Reject') {
                return redirect()->route('leave.index')->with('error', __('Team Lead can only set status to TL Approved or Reject.'));
            }
            $leave->status = $request->status;
        } elseif ($user->type == 'Reporting Manager') {
            if ($request->status != 'Approved' && $request->status != 'Reject') {
                return redirect()->route('leave.index')->with('error', __('Reporting Manager can only set status to Approved or Reject.'));
            }
            $leave->status = $request->status;
        }

        if ($leave->status == 'Approved') {
            $startDate = new \DateTime($leave->start_date);
            $endDate = new \DateTime($leave->end_date);
            $endDate->add(new \DateInterval('P1D'));
            $leave->total_leave_days = $startDate->diff($endDate)->days;
        }

        $leave->save();

        // Notify next approver (Reporting Manager) after TL approval
$settings = Utility::settings($user->creatorId());

if (
    $leave->status == 'tl_approved' &&
    !empty($employee->reporting_manager) &&
    isset($settings['new_leave']) &&
    $settings['new_leave'] == 1
) {
    // Safely extract email from "Reporting Manager (rm@example.com)"
    if (preg_match('/\(([^)]+)\)/', $employee->reporting_manager, $matches)) {
        $managerEmail = trim($matches[1]);
    } else {
        $managerEmail = trim($employee->reporting_manager);
    }

    

    $manager = Employee::where('email', $managerEmail)->first();

    if ($manager) {
        $manager_user = User::find($manager->user_id);

        if ($manager_user) {
            $leaveArr = [
                'employee_name' => $employee->name,
                'start_date' => $leave->start_date,
                'end_date' => $leave->end_date,
                'leave_reason' => $leave->leave_reason,
                'user_name' => $user->name, // optional - can be used in email template
            ];

            // 🔔 Send 'new_leave' email to reporting manager
            Utility::sendEmailTemplate('new_leave', [$manager_user->id => $manager_user->email], $leaveArr);
         

            // 📲 Send WhatsApp notification
            if (!empty($manager->phone)) {
                $number = preg_replace('/[^0-9]/', '', $manager->phone);
                $message = "Hello {$manager->name},\n"
                    . "A leave request from {$employee->name} has been approved by TL and needs your review.\n"
                    . "Reason: {$leave->leave_reason}\n"
                    . "From: {$leave->start_date} To: {$leave->end_date}.";
                Utility::sendWhatsappMessage($number, $message);
               
            }

        } else {
            \Log::warning("Reporting Manager User Not Found: $managerEmail, triggered by user ID: " . $user->id);
        }
    } else {
        \Log::warning("Reporting Manager Employee Not Found: $managerEmail, triggered by user ID: " . $user->id);
    }
}

        
      elseif (
    in_array($leave->status, ['Approved', 'Reject']) &&
    isset($settings['leave_action_sent']) &&
    $settings['leave_action_sent'] == 1
) {
    // Optional: Ensure only Reporting Manager triggers this
    $reportingManagerEmail = '';
    if (preg_match('/\(([^)]+)\)/', $employee->reporting_manager, $matches)) {
        $reportingManagerEmail = trim($matches[1]);
    } else {
        $reportingManagerEmail = trim($employee->reporting_manager);
    }

    $isManager = $user->email === $reportingManagerEmail;

    if ($isManager) {
        $leaveArr = [
            'leave_name' => $employee->name,
            'leave_status' => $leave->status,
            'leave_reason' => $leave->leave_reason,
            'leave_start_date' => $leave->start_date,
            'leave_end_date' => $leave->end_date,
            'total_leave_days' => $leave->total_leave_days,
        ];

        $resp = Utility::sendEmailTemplate('leave_action_sent', [$employee->user_id => $employee->email], $leaveArr);
        if (!$resp['is_success']) {
            \Log::error('Email Sending Failed for Leave Action: Leave ID ' . $leave->id . ', Error ' . ($resp['error'] ?? 'Unknown error') . ', User ID ' . $user->id);
        }

        if (!empty($employee->phone)) {
            $number = preg_replace('/[^0-9]/', '', $employee->phone);
            $statusText = $leave->status == 'Approved' ? 'approved' : 'rejected';
            $message = "Dear {$employee->name},\n"
                . "Your leave request has been *{$statusText}*.\n"
                . "Reason: {$leave->leave_reason}\n"
                . "From: {$leave->start_date} To: {$leave->end_date}\n"
                . "Total Days: {$leave->total_leave_days}.";
            Utility::sendWhatsappMessage($number, $message);
        }
    }



            return redirect()->route('leave.index')->with('success', __('Leave status successfully updated.') . ($resp['is_success'] ? '' : '<br> <span class="text-danger">' . ($resp['error'] ?? 'Email sending failed.') . '</span>'));
        }

        return redirect()->route('leave.index')->with('success', __('Leave status successfully updated.'));
    }

    public function jsoncount(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $leave_counts = [];
        $leave_types = LeaveType::where('created_by', Auth::user()->creatorId())->get();
        $year = date('Y');

        foreach ($leave_types as $type) {
            $counts = Leave::selectRaw('
                COALESCE(SUM(
                    CASE
                        WHEN YEAR(start_date) = ? AND YEAR(end_date) = ? THEN total_leave_days
                        WHEN YEAR(start_date) = ? THEN DATEDIFF(LAST_DAY(start_date), start_date) + 1
                        WHEN YEAR(end_date) = ? THEN DATEDIFF(end_date, DATE_FORMAT(end_date, "%Y-01-01")) + 1
                        ELSE 0
                    END
                ), 0) AS total_leave
            ')
                ->where('leave_type_id', $type->id)
                ->where('employee_id', $request->employee_id)
                ->where('status', '!=', 'Reject')
                ->where(function ($query) use ($year) {
                    $query->whereYear('start_date', $year)
                          ->orWhereYear('end_date', $year);
                })
                ->addBinding([$year, $year, $year, $year], 'select')
                ->first();

            $leave_counts[] = [
                'total_leave' => $counts->total_leave ?? 0,
                'title' => $type->title,
                'days' => $type->days,
                'id' => $type->id,
            ];
        }

        return response()->json($leave_counts);
    }
}